

//drawglwidget.h
//(c) 2018 The University of Aizu
//This software is released under the GNU General Public License.


#ifndef DRAWGLWIDGET_H
#define DRAWGLWIDGET_H

#define DELTA 1/512
#define EPSILON 0.000001
#define STRLEN 100
#define BCVLEN 5
#define ROOM 10
#define THRESHOLDFOV 0.01

#include <QtOpenGL>
#include <QGLWidget>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <iostream>
#include <QtGui>
#include <QWidget>
#include <QPair>
#include <GLUT/glut.h>
#include <opencv/cv.hpp>
#include <opencv/highgui.h>
#include <vector>
#include <QFileInfo>
#include <fitsio.h>
#include <CCfits/CCfits>
#include "drawSpacecraftModel.h"
#include "drawTargetModel.h"
#include "computeelements.h"

#include "SpiceUsr.h"

#ifdef __APPLE__
  #include <OpenCL/opencl.h>
  #include <OpenCL/cl.h>
  #include <cl.hpp>
#else
  #include <CL/cl.h>
#endif

using namespace std;
using namespace cv;

using namespace CCfits;

class instParam{
public:
    QString instName, instID;
    QString forcalLength;
    SpiceDouble boundaryCorners[4][3];
    GLfloat projection[16];
    GLfloat frstum[4];
    QString FOVAngle, instType;
    QString colorName;
    GLboolean isDrawFootprint;
    QColor colorCode;

public:
    instParam(){

    }

    instParam(QString instName, SpiceInt instID, SpiceDouble forcalLength, SpiceDouble boundaryCorners[4][3], SpiceDouble FOVAngle, GLfloat projection[16], QString colorName, QColor colorCode){
        this->instName=instName;
        this->instID=QString::number(instID);
        this->forcalLength=QString::number(forcalLength);
        this->FOVAngle=QString::number(FOVAngle);
        this->colorName=colorName;
        this->colorCode=colorCode;
        this->isDrawFootprint=GL_TRUE;
        
        memcpy(this->boundaryCorners, boundaryCorners, sizeof(SpiceDouble)*12);
        memcpy(this->projection, projection, sizeof(GLfloat)*16);

        if(this->FOVAngle.toDouble()>THRESHOLDFOV){
            instType="FOV";
        }
        else{
            instType="Boresight";
        }

        qDebug() << instName;
    }

    ~instParam(){

    }
    void setIsDrawFootprint(SpiceBoolean isDrawFootprint){
        this->isDrawFootprint=isDrawFootprint;
    }

    QString getInstName() const{
        return instName;
    }

    void getBoundaryCorner(SpiceDouble boundaryCorners[][3]) const{
        memcpy(boundaryCorners, this->boundaryCorners, sizeof(SpiceDouble)*12);
    }

    void getProjection(GLfloat projection[16]) const{
        memcpy(projection, this->projection, sizeof(GLfloat)*16);
    }

    SpiceDouble getFOVAngle() const{
        return FOVAngle.toDouble();
    }
    SpiceDouble getFocalLength() const{
        return forcalLength.toDouble();
    }

    QString getInstrumentType() const{
        return instType;
    }

    SpiceBoolean getIsDrawFootprint() const{
        return isDrawFootprint;
    }

    QColor getFootprintColorCode() const{
        return colorCode;
    }

    QString getFootprintColorName() const{
        return colorName;
    }
};

class spacecraftParam{

public:
    QVector<instParam> instParamsOnSpacecraft;
    QString SCName, SCID;
    QString var_name;
    SpiceInt n_return;
    SpiceDouble forcalLength;
    SpiceDouble boundaryCorners[4][3];
    GLfloat projection[16];
    SpiceBoolean foundInst;
    SpiceDouble ang1, ang2, mag;
    SpiceDouble untbnd[4][3];
    SpiceDouble bsght[3], vec1[3], vec2[3];
    SpiceDouble FOVAngle;
    QVector<QPair<QString, QVector4D> > footprintsColors;
    QList<QString> keys;
    QString colorName, key;
    QColor colorCode;
    QVector4D c;

    spacecraftParam(){

    }

    spacecraftParam(QString SCName){
        
        SpiceBoolean found;
        SpiceInt id;
        bodn2c_c(SCName.toStdString().c_str(), &id, &found);

        this->SCName=SCName;
        this->SCID=id;
        footprintsColors.append(QPair<QString, QVector4D>("Red", QVector4D(1.0, 0.0, 0.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Green", QVector4D(0.0, 1.0, 0.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Blue", QVector4D(0.0, 0.0, 1.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Cyan", QVector4D(0.0, 1.0, 1.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Magenta", QVector4D(1.0, 0.0, 1.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Yellow", QVector4D(1.0, 1.0, 0.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Orange", QVector4D(1.0, 0.56, 0.0, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Purple", QVector4D(0.58, 0.0, 0.83, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("Pink", QVector4D(1.0, 0.41, 0.71, 1.0)));
        footprintsColors.append(QPair<QString, QVector4D>("YellowGreen", QVector4D(0.68, 1.0, 0.18, 1.0)));

    }

    ~spacecraftParam(){

    }

    void setInstParam(QString instName){
        
        SpiceBoolean found;
        SpiceInt instId;
        QString instIdNum;
        bodn2c_c(instName.toStdString().c_str(), &instId, &found);

        if(instId>0){
            printf("inst error");
        }
        if(!found){
           printf("setInstParam Instrument name %s could not be translated to an ID code\n", instName.toStdString().c_str());
        }
        instIdNum=QString(QString::number(instId));

        var_name="INS"+instIdNum+"_FOCAL_LENGTH";
        
        gdpool_c(var_name.toStdString().c_str(), 0, ROOM, &n_return, &forcalLength, &foundInst);
        

        var_name="INS"+instIdNum+"_FOV_BOUNDARY_CORNERS";
        gdpool_c(var_name.toStdString().c_str(), 0, 12, &n_return, &boundaryCorners[0][0], &foundInst);

        var_name="INS"+instIdNum+"_BORESIGHT";
        gdpool_c(var_name.toStdString().c_str(), 0, ROOM, &n_return, bsght, &foundInst);

        for(int i=0; i<4; i++){
            boundaryCorners[i][2]=forcalLength;
        }

        SpiceDouble bc[4][3];
        memcpy(bc, boundaryCorners, sizeof(boundaryCorners));
        for(int i=0; i<4; i++){
            for(int j=0; j<2; j++){
                bc[i][j]=round(boundaryCorners[i][j]*1000.0)/1000.0;
            }
        }

        unorm_c(&(boundaryCorners[0][0]), &(untbnd[0][0]), &mag);
        unorm_c(&(boundaryCorners[1][0]), &(untbnd[1][0]), &mag);
        unorm_c(&(boundaryCorners[2][0]), &(untbnd[2][0]), &mag);
        vadd_c(&(untbnd[0][0]), &(untbnd[1][0]), vec1);
        vscl_c(0.5, vec1, vec1);
        vadd_c(&(untbnd[1][0]), &(untbnd[2][0]), vec2);
        vscl_c(0.5, vec2, vec2);

        ang1=vsep_c(bsght, vec1);
        ang2=vsep_c(bsght, vec2);
        ang1*=RadToDeg;
        ang2*=RadToDeg;
        ang1=qCeil(ang1*100.0)/100.0;
        ang2=qCeil(ang2*100.0)/100.0;

        if(ang1>ang2){
            FOVAngle=ang1*2.0;
        }
        else{
            FOVAngle=ang2*2.0;
        }
        

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        glFrustum(boundaryCorners[1][0], boundaryCorners[0][0], boundaryCorners[3][0], boundaryCorners[2][0], forcalLength, 1000);
        glGetFloatv(GL_PROJECTION_MATRIX, projection);
        colorName=footprintsColors.at(instParamsOnSpacecraft.length()).first;
        
        
        c=footprintsColors.at(instParamsOnSpacecraft.length()).second;
        colorCode.setRgbF(c.x(), c.y(), c.z(), c.w());
        

        instParam instparam=instParam(instName, instId, forcalLength, boundaryCorners, FOVAngle, projection, colorName, colorCode);
        instParamsOnSpacecraft.append(instparam);

    }

    
    SpiceBoolean isInstRegistered(QString instName){
        SpiceBoolean registered=SPICEFALSE;
        SpiceBoolean found;
        SpiceInt instId;
        bodn2c_c(instName.toStdString().c_str(), &instId, &found);

        
        
        if(!found){
            cout<<"not"<<endl;
            return SPICETRUE;
        }
        for(int i=0; i<instParamsOnSpacecraft.length(); i++){
            QString name=instParamsOnSpacecraft.at(i).getInstName();
            if(name==instName){
                registered=SPICETRUE;
                break;
            }
        }
        return registered;
    }

    void setIsDrawFootprint(QString instName, SpiceBoolean isdraw){
        SpiceInt id=getParamSCInstID(instName);
        instParam instparam=instParamsOnSpacecraft.at(id);
        instparam.setIsDrawFootprint(isdraw);
        instParamsOnSpacecraft[id]=instparam;
    }

    SpiceInt getRegisteredInstNum(){
        
        return instParamsOnSpacecraft.length();
    }

    SpiceInt getParamSCInstID(QString instName){
        SpiceInt id=0;
        

        for(int i=0; i<instParamsOnSpacecraft.length(); i++){
            if(instParamsOnSpacecraft.at(i).getInstName()==instName){
                
                id=i;
            }
        }
        
        
        
        return id;
    }

    SpiceDouble getForcalLengthAndBoundaryCorners(QString instName, SpiceDouble boundaryCorner[][3]){
        SpiceInt id=getParamSCInstID(instName);
        instParamsOnSpacecraft.at(id).getBoundaryCorner(boundaryCorner);

        return instParamsOnSpacecraft.at(id).getFocalLength();
    }

    SpiceDouble getFOVAngle(QString instName){
         SpiceInt id=getParamSCInstID(instName);
         return instParamsOnSpacecraft.at(id).getFOVAngle();
    }

    QString getInstType(QString instName){
         SpiceInt id=getParamSCInstID(instName);
         return instParamsOnSpacecraft.at(id).getInstrumentType();
    }

    QColor getFootprintColorCode(QString instName){
         SpiceInt id=getParamSCInstID(instName);
         return instParamsOnSpacecraft.at(id).getFootprintColorCode();
    }

    QString getFootprintColorName(QString instName){
         SpiceInt id=getParamSCInstID(instName);
         return instParamsOnSpacecraft.at(id).getFootprintColorName();
    }

    void getProjectionMatrix(QString instName, GLfloat projection[16]){
        SpiceInt id=getParamSCInstID(instName);
        instParam instparam=instParamsOnSpacecraft.at(id);
        instparam.getProjection(projection);
    }
};

class drawGLWidget : public QGLWidget
{
    Q_OBJECT
public:
    explicit drawGLWidget(QWidget *parent = 0);
    ~drawGLWidget();
    virtual void initializeGL();
    virtual void resizeGL(int width, int height);
    virtual void paintGL();
    QString getfileName();
    void draw3Dperspective();
    void drawGeo();
    void compSwing();

    void drawInstName(QString);

    void drawOverlayImage(QString);
    void mousePressEvent(QMouseEvent *event);
    void mouseMoveEvent(QMouseEvent *event);
    void keyPressEvent(QKeyEvent *event);
    void timerEvent(QTimerEvent *event);
    void wheelEvent(QWheelEvent *event);
    void drawVec(SpiceDouble[3], SpiceDouble[3], SpiceDouble[3], GLboolean, GLboolean);
    void drawCenterofFOV();
    void drawOrbital();

    void drawShadow();
    SpiceInt compSubSCPolygon();
    void compBoundingSphere();
    GLboolean crossingDetection(const QString, const QColor, GLboolean);
    QMatrix4x4 calculateMatrixInv(const GLfloat* mat);
    void calculateAffineTranslate();
    void compSphereMember();
    GLboolean MygluUnProject(const QString, const float* const, const GLfloat* const, const GLint* const);
    void prepareFootprint();
    int initCL();
    
    void draw_hapke();

    void visualizeFrustum();

private:
    QVector<spacecraftParam> SCParameters;
    QVector<QString> HARMONICSModes, instrumentCodes;
    QString savefilename;
    QString SCName, SCCode;
    QString targetFrame;
    QString instName;
    QString targetName;
    QVector<QString> instNames;
    SpiceChar startUTC[STRLEN], endUTC[STRLEN];
    SpiceChar utc[STRLEN];
    SpiceDouble start_et, end_et, et, preET;
    SpiceDouble target_pos[3], sc_pos[3], posSC[3], bsightDir[3], upVec[3], earth_pos[3], sun_pos[3];
    SpiceDouble inputXpos, inputYpos, inputZpos, preX, preY, preZ;
    SpiceDouble roll, pitch, yaw, dis, preRoll, prePitch, preYaw;
    SpiceDouble interval, rollUL, rollLL, rollVariation, pitchUL, pitchLL, pitchVariation, yawUL, yawLL, yawVariation;
    SpiceDouble rollCurrentAngle, pitchCurrentAngle, yawCurrentAngle;
    GLdouble rollmid, pitchmid, yawmid;
    GLdouble r, p, y;
    SpiceDouble SCBoreSight[3];
    SpiceDouble inputRoll, inputPitch, inputYaw;
    SpiceDouble up_instFromInput[3];
    SpiceDouble lt, lt_earth, lt_target;
    QVector3D earthVec, sunVec, bsightVec;
    SpiceDouble phase, incidence, emission;
    SpiceInt subPolygonID;
    SpiceBoolean foundIll;
    SpiceDouble step;
    SpiceInt instId;
    SpiceChar shape[STRLEN];
    SpiceChar frame[STRLEN];
    SpiceDouble bsight[3];
    SpiceDouble up_inst[3];
    SpiceDouble defaultBoundaryCorners[5];
    SpiceDouble rotateSC[3];
    SpiceBoolean inputxyz;
    SpiceDouble compBoresight[3];
    SpiceFloat targetRot[16];
    QMatrix4x4 affine;

    SpiceDouble rotateX1[3];
    SpiceDouble rotateY1[3];
    SpiceDouble rotateZ1[3];

    SpiceDouble rotateX2[3];
    SpiceDouble rotateY2[3];
    SpiceDouble rotateZ2[3];

    GLfloat observerRotate[3];
    GLfloat targetRotate[3];
    GLfloat observerTranslate[3];
    GLfloat targetTranslate[3];
    GLfloat SCPosTranslate[3];
    GLfloat geometryTranslate[3];
    GLfloat geometryRotate[3];
    static const GLdouble genfunc[][4];


    SpiceDouble bounds[4][3];
    GLfloat FOVLength;

    QPoint lastPos;
    DrawTargetModel targetModel;
    DrawSpacecraftModel spacecraftModel;
    computeElements compElements;
    QString harmonicsMode;
    QString glViewMode;
    GLboolean isChangeVal, isScanning, isAutoDrawFootprints, drewFootprint, isComputedSubpolygon;
    QBasicTimer *timer;
    GLboolean animation, initializedFootprint, isRestoredShadowData, isOnceRestore;
    GLboolean isComputeSubPolygon, drawSunVec, drawEarthVec, isDrawCenterOfFOV, drawOverlay, drawAreaOfFOV, drawOrbit, drawFootprints, isDrawShadow, drewShadow;
    QVector<GLboolean> drawScientificInstruments;
    QList<QVector3D> orbitList;
    QString readedfilename;
    GLboolean logFileCreate;
    QImage loadedImageFile;
    QString loadedFiletype;
    GLfloat projection[16], subPointProjection[16];
    GLfloat modelView[16]; 
    GLint viewport[4];    
    GLboolean isRecordSubPointProjection;
    QMatrix4x4 proMat, proInv, modelMat, modelInv, viewportInv;
    QVector4D clickPos;
    QVector4D nearPos, farPos, rayVec, rayVecUnit;
    QVector<QVector4D> translatedVertex, screenStart, rayDirection;
    QHash<QString, QVector4D> footprintsColor;
    QVector<QVector4D> spheresCenter;
    QVector<QVector4D> FOVArea, boresightArea, subSCPointPolygonArea;
    QMatrix4x4 affineMatrix, SCAffineMatrix;
    QString rollDir, pitchDir, yawDir;
    QTextCodec* codec;
    QList<GLuint> polygonIDInObservedArea;
    QList<GLboolean> shadowFlag;

    bool isNIRS, isLIDAR;


    QVector<cl::Platform> platforms;
    vector<cl::Platform> stdPlat;
    QVector<cl::Device> devices;
    vector<cl::Device> stdDevice;

    cl::Context context_CompRayDir;
    cl::Context context_ParallelCollision;
    cl::Context context_AffineTranslate;
    cl::Context context_DrawShadows;
    cl::Context context_CompShpereMember;

    cl::Program::Sources source_CompRayDir;
    cl::Program::Sources source_ParallelCollision;
    cl::Program::Sources source_AffineTranslate;
    cl::Program::Sources source_DrawShadows;
    cl::Program::Sources source_CompShpereMember;

    cl::Program program_CompRayDir;
    cl::Program program_ParallelCollision;
    cl::Program program_AffineTranslate;
    cl::Program program_DrawShadows;
    cl::Program program_CompShpereMember;

    cl::CommandQueue q_CompRayDir;
    cl::CommandQueue q_ParallelCollision;
    cl::CommandQueue q_AffineTranslate;
    cl::CommandQueue q_DrawShadows;
    cl::CommandQueue q_CompShpereMember;

    cl::Kernel kernel_CompRayDir;
    cl::Kernel kernel_ParallelCollision;
    cl::Kernel kernel_AffineTranslate;
    cl::Kernel kernel_DrawShadows;
    cl::Kernel kernel_CompShpereMember;

    QVector<QVector4D> WandH;
    QVector<QVector4D> ONC_TofFOV, TIRofFOV, LIDARandNIRS3ofFOV, otherArea;
    GLuint screenWidth, screenHeight;
    QVector<GLuint> scientificInstrumentsPeriod;
    QColor dummyColor;
    QString kernelName;

signals:
    void setUTC(QString);
    void finishedAnim_signal();
    void notFoundInst_signal();
    void setWindowTime_signal(SpiceDouble);
    void setWindowPos_signal(SpiceDouble, SpiceDouble, SpiceDouble);
    void setWindowDisTargettoSC_signal(SpiceDouble);
    void setWindowDisEarthtoSC_signal(SpiceDouble);
    void setWindowBoresight_signal(SpiceDouble, SpiceDouble, SpiceDouble);
    void setWindowPosAndAtt_signal(SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble);
    void setWindowIll_signal(SpiceDouble, SpiceDouble, SpiceDouble, SpiceBoolean);
    void changeKeyStroke_signal(SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble);
    void setInstrumentsParameters_signal(QString,QString,QString,QString,QString,QColor);
    void getResolution_signal(QString);
    void drawFootprints_signal();
    void getPolygonID_signal(QList<GLuint>, QList<GLboolean>);
    void getSubSCPoint_signal(SpiceInt);

public slots:
    void viewMode_slot(QString);
    void saveImage_slot();
    void startAnimation();
    void stopAnimation();
    void setMode_slot(QString);
    void setIsChangeVal_slot(GLboolean);
    void setSC_slot(QString);
    void setFrame_slot(QString);
    void setInst_slot(QString);
    void setTarget_slot(QString);
    void setTime_slot(SpiceDouble, SpiceDouble, SpiceDouble);
    void setInputPosAndAtt_slot(SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble);

    void sendInstruments_slot(QString,QStringList);
    void stepChanged_slot(GLint);
    void setReadedfile_slot(QString);
    void setLoadSCModelFile_slot(QString);
    void setLoadTargetModelFile_slot(QString);
    void setLoadSCMTLFile_slot(QString);
    void setLoadTargetMTLFile_slot(QString);
    void isComputeSubPolygon_slot(GLboolean);
    void drawSunVec_slot(GLboolean);
    void drawEarthVec_slot(GLboolean);
    void drawCenterOfFOV_slot(GLboolean);
    void drawOverlay_slot(GLboolean);
    void drawAreaOfFOV_slot(GLboolean);
    void drawOrbit_slot(GLboolean);
    void sendPolygonID_slot();
    void createLogFile_slot(GLboolean);
    void loadcontorImage_slot(QImage, QString);
    void isDrawFootprints_slot(GLboolean);
    void drawFootprintsOnTarget_slot();


    void setIsDrawFootprint_signal(QString, SpiceBoolean);


    void initializeFootprintsInfo_slot();
    void isDrawShadow_slot(GLboolean);    
    void getResolutionOfModel_slot();
    void visualizationObservedData_slot(QStringList);
    void setSwingParameters_slot(SpiceInt, SpiceDouble,SpiceDouble,SpiceDouble,SpiceDouble,SpiceDouble,SpiceDouble, SpiceDouble, SpiceDouble, SpiceDouble);
    void setAutoDrawFootprintsDuringScan_slot(GLboolean);
    void mapTexture_slot(QString, QString); 
    void clearTexture_slot();
    

    void draw_NIRS_LINE_slot(bool);
    void draw_LIDAR_LINE_slot(bool);

    DrawTargetModel* getTargetModel_slot();
    void setFOVLength_slot(GLfloat);
};

#endif 
